/* Program created by Douglas Stockman.  This is a demo program that
   provides examples of how to create PostScript, HP-GL/2, and DXF
   (wireframe only) structured drawing files.  The program was compiled
   using SAS/C V6.3 on an Amiga 3000, AmigaDos 2.04.
   
   Please note, I did not have an HP-GL/2 capable plotter available
   for testing this code (the HP loaner needed to be returned before
   this article was written).  HP-GL/2 does not intrinsically support
   hidden line removal and does not have the ability to erase graphic
   elements by filling a shape with background color.  Therefore, if
   the user wants to support HP-GL/2 hidden line removal in their code,
   they will need to write the required routines.
   
   I have included the most basic wireframe-only code for DXF support.
   This was briefly tested using TurboCAD.  The interested programmer
   can use the supplied code to create more robust DXF code.  I have
   included a DXF init file. This init file preceeds any code-generated
   output.
   
   Must copy the following files to the S: directory, or change in the
   code where to look for files.
      s:sd.PSinit
      s:sd.DXFinit
   
   
   Amiga, AmigaDOS, registered trademarks of CBM
   PostScript, registered trademark of Adobe Systems
   HP-GL/2, registered trademark Hewlett Packard Corp.
   DXF, created by AutoDesk
   TurboCAD, registered trademark by IMSI
*/

#include <stdio.h>      /* standard ANSI C includes */
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <ctype.h>

#include <exec/types.h>    /* Amiga specific includes */
#include <exec/libraries.h>
#include <intuition/intuition.h>
#include <libraries/dos.h>
#include <dos/dos.h>
#include <dos/dostags.h>
#include <intuition/screens.h>
#include <graphics/gfxbase.h>
#include <graphics/text.h>
#include <graphics/gfxmacros.h>
#include <utility/tagitem.h>
#include <graphics/displayinfo.h>

#include <clib/exec_protos.h>       /* Amiga specific protos */
#include <clib/graphics_protos.h>
#include <clib/diskfont_protos.h>
#include <clib/intuition_protos.h>
#include <clib/alib_protos.h>
#include <clib/dos_protos.h>
#include <clib/layers_protos.h>

/* functions */
void main(void);
BOOL Open_Libs(void);
void Close_Libs(void);
BOOL Open_ScrWin(void);
void Close_ScrWin(void);
void draw_figs(void);
void get_font(void);
void sd_Text(UBYTE *draw_text, UBYTE draw_color);
void sd_Move(float x_loc, float y_loc);
void sd_Draw(UBYTE pat, UBYTE color, UBYTE l_thick,
            float x_loc, float y_loc);
void sd_DrawLine(UBYTE pat, UBYTE color, UBYTE l_thick,
                float x1,float y1,float x2, float y2);
void sd_Box(float x1,float y1,float x2,float y2,int l_thick, int f_col,
   int f_pat, int l_col, int l_pat, BOOL fill, BOOL outline);
BOOL sd_DrawSlice2(float xloc, float yloc, float rad, float angstart,
        float angend, int l_thick, int f_pat, int f_col,
        int l_col, int l_pat, BOOL fill, BOOL outline);
void arc(float xloc, float yloc, float rad, float ang1, float ang2,
   int l_thick, int l_col, int l_pat);
void ps_colorout(ULONG d_color);
void ps_fillout(int d_pat);
void split_colors(UBYTE color, ULONG *red, ULONG *green, ULONG *blue);
void loc_pieline(float radius, float angle, float aspect, float *xt, float *yt);
BOOL sd_AreaDraw2(float *x, float *y, int pt_sz, int f_col, int l_col, int f_pat,
   int l_thick, int l_pat, BOOL fill, BOOL outline);
UWORD *sg_AreaBuffer(int pt_sz);
BOOL set_pats(void);
void hp_fillpat(UBYTE color, UBYTE pattern);

/* used to define where output is sent */
#define SCREEN  0
#define POST    1
#define HPGL2   2
#define DXF     3

#define DTOR  0.017453     /* converts degrees to radians: x * DTOR=radians */

/*------ Global Variables --------------*/
UBYTE Output_Device=SCREEN;
struct Screen *sd_scr = NULL;
struct Window *sd_win = NULL;
struct RastPort *sd_rp = NULL;
struct GfxBase *GfxBase=NULL;
struct IntuitionBase *IntuitionBase=NULL;
struct Library *DiskfontBase=NULL;
struct ColorMap *ssgcm;
struct ViewPort *ssgvp;
struct AreaInfo areaInfo;

struct TextAttr Topaz80 =
{
    "topaz.font",	/* Name */
    8,			/* YSize */
    0,			/* Style */
    0,			/* Flags */
};

struct TextFont *tFont = NULL, *font=NULL; 
struct TextAttr tAttr;

PLANEPTR planePtr = NULL;
struct TmpRas tmpRas;

float PS_Width=8.0, PS_Height=5.0, hp_scalex=10.0, hp_scaley=8.0;
float hp_offsetx=0.0, hp_offsety=0.0;
float dxfX=0.0, dxfY=0.0, dxfWIDE=11./640., dxfHIGH=8.5/380.;
ULONG W_Width=640, W_Height=400, tmpwidth, tmpheight;
int i=0, hp_orient=1;
int F_Size=18, F_Style=0, data_thick=1;
UBYTE Output_File_Name[150], F_Name[150], ps_orient=0;/*, FillPat=0, LinePat=1;*/
UBYTE Pattern_PS[16][10];  /* fill pattern for PostScript */
UBYTE DXFLT[9][12]={"CONTINUOUS","DASHED","HIDDEN","CENTER","PHANTOM","DOT",
                 "DASHDOT","BORDER","DIVIDE"};
UWORD pattern[10][4];   /* Fill patterns for AmigaDos */
UWORD lpattern[10];     /* line patterns for AmigaDos */
UBYTE *ps_lpat[10];     /* line patterns for PostScript */

BOOL Color_PS=FALSE;    /* Determines if want to output PS color information */
                        /* as gray level or RGB level */
FILE *Outfp=NULL, *Infp=NULL;

/*--------------------------------------*/
#define MAP_X(x)    (10 + (x))     
#define MAP_Y(y)    (384 - (y))    /* flip Y coordinates so origin at bottom */
                                 /* of screen */
/* use PSX2 for calculating radius */
#define MAP_PSX2(x)  ((PS_Width)/W_Width * (x))
#define MAP_PSY2(y)  ((PS_Height)/W_Height)*(W_Height-(y))

/* Convert from screen coordinates to PostScript coordinates */
#define MAP_PSX(x)  (0.25 + (PS_Width/W_Width * (x)))
#define MAP_PSY(y)  (0.25 + (PS_Height/W_Height)*(W_Height-(y)))

void main(void){
   char c=NULL;
   int color_max=8;    /* used to set number of colors for HPGL2 */
   
   if(Open_Libs()){
      if(Open_ScrWin()){
            /* send output to window */
            Output_Device = SCREEN;
            draw_figs();
            Delay(50 * 5);     /* delay 5 seconds */
            
            /* send output to PS device */
            strcpy(Output_File_Name, "ram:tmp.ps");
            if(Outfp = fopen(Output_File_Name, "w")){
               Output_Device = POST;
               
               /* copy PS init file to output file */
               if(Infp=fopen("s:sd.PSinit", "r")){
                  while((c=fgetc(Infp)) != EOF) fputc(c, Outfp);
                  fclose(Infp);
                  draw_figs();
                  fprintf(Outfp, "showpage\n");
               }
               else puts("Error getting PostScript init file\n");
               fclose(Outfp);
               Delay(50 * 5);
            }
            
            /* send output to HPGL2 device */
            strcpy(Output_File_Name, "ram:tmp.hpgl");
            if(Outfp=fopen(Output_File_Name, "w")){
                Output_Device = HPGL2;
                
                /*Initialize Plotter & Set Size*/
                fprintf(Outfp,"%c%sNP%d",27,"%-1BBPINPS", color_max);
                
                fprintf(Outfp,"PW0.35");     /** Pen widths **/
                
                /* If desired, Rotate to Portrait **/
                if(hp_orient==0)  fprintf(Outfp,"RO90"); 
                
                /*Set Size and Location of Plot*/
                fprintf(Outfp,"IP%d,%d,%d,%d",(int)(hp_offsetx*1016.),
                   (int)(hp_offsety*1016.),(int)((hp_scalex+hp_offsetx)*1016.),
                   (int)((hp_scaley+hp_offsety)*1016.));
                  
                /* set scale of plot to use screen coordinates */
                fprintf(Outfp,"SC%d,%d,%d,%d", 0,W_Width,W_Height,0);
        
                draw_figs();
                
                /* end plot, append RP%d; if want copies */
                fprintf(Outfp, "PUSP0PG;");

                fclose(Outfp);
                Delay(50 * 5);
            }
            
            
            /* send output to DXF files.  **/
            strcpy(Output_File_Name, "ram:tmp.dxf");
            if(Outfp=fopen(Output_File_Name, "w")){
                Output_Device = DXF;
                
                /* copy DXF init file to output file */
                if(Infp=fopen("s:sd.DXFinit", "r")){
                   while((c=fgetc(Infp)) != EOF) fputc(c, Outfp);
                   fclose(Infp);
                }
                draw_figs();
                
                /* terminate DXF instructions */
                fputs("0\nENDSEC\n0\nEOF\n", Outfp);

                fclose(Outfp);
                Delay(50 * 5);
            }
            Close_ScrWin();
         }
         Close_Libs();
      }
    exit(0);
}
/*-------------------------------------------------------*/
/* Open Amiga custom screen and window. Set colors */
BOOL Open_ScrWin(void){
   int i=0;
   UBYTE   palcolor[4][3];   
   
   /* define default screen colors */
   palcolor[0][0]=15;   palcolor[0][1]=15;   palcolor[0][2]=15;
   palcolor[1][0]=0;    palcolor[1][1]=0;    palcolor[1][2]=0;
   palcolor[2][0]=8;    palcolor[2][1]=0;    palcolor[2][2]=8;
   palcolor[3][0]=10;   palcolor[3][1]=10;   palcolor[3][2]=10;

   /* open screen */
   if(!(sd_scr = OpenScreenTags(NULL,
      SA_Width, W_Width, SA_Height, W_Height,
      SA_Depth,(ULONG)2, SA_DisplayID,  HIRESLACE_KEY,
      SA_Title, "StructDraw.c",
      SA_Font, &Topaz80, SA_Type, CUSTOMSCREEN, TAG_END))){
         puts("Cannot open screen, ABORTING!");
         return(FALSE);
   }
   
   for(i=0;i<4;i++){    /* set screen colors */
      SetRGB4(&sd_scr->ViewPort, i, palcolor[i][0],
         palcolor[i][1], palcolor[i][2]);
   }

   /* open window */
   if(!(sd_win = OpenWindowTags(NULL,   
      WA_Left, 0, WA_Top, 12,
      WA_Width, W_Width, WA_Height,W_Height-12,
      WA_CloseGadget, FALSE, WA_DragBar, TRUE,
      WA_SmartRefresh, TRUE, WA_Activate, TRUE,
      WA_Title,"StructDraw Window",
      WA_CustomScreen, sd_scr, TAG_END))){
         puts("Cannot open window, ABORTING!");
         CloseScreen(sd_scr);
         sd_scr = NULL;
         return(FALSE);
   }
    GT_RefreshWindow(sd_win, NULL);

   sd_rp = sd_win->RPort;
   ssgvp = &(sd_scr)->ViewPort;
   ssgcm = ssgvp->ColorMap;

   /** create temp raster for Flood/Area fills **/
   if(planePtr)    FreeRaster(planePtr, tmpwidth, tmpheight);
   tmpheight = W_Height + 50;
   tmpwidth  = W_Width + 50;
   planePtr = (PLANEPTR) AllocRaster(tmpwidth, tmpheight);
   if(planePtr){
      InitTmpRas(&tmpRas, planePtr, RASSIZE( tmpwidth, tmpheight));
      sd_rp->TmpRas = &tmpRas;
   }
   else{
      puts("Cannot Allocate TmpRas, SHUTTING DOWN!");
      if(sd_win)  CloseWindow(sd_win);
      if(sd_scr)  CloseScreen(sd_scr);
      return(FALSE);
   }
   return(TRUE);
}
/*----------------------------------------------------*/
/* Open Amiga-required libraries */
BOOL Open_Libs(void){

   if(!(GfxBase = (struct GfxBase *)
	   OpenLibrary("graphics.library", 36L))){
   	   puts("cannot open graphics library 36+\n");
         return(FALSE);
   }

   if(!(IntuitionBase = (struct IntuitionBase *)
	   OpenLibrary("intuition.library", 36L))){
	      puts("cannot open intuition library 36+");
         CloseLibrary((struct Library *)GfxBase);
         GfxBase=NULL;
         return(FALSE);
   }

   if(!(DiskfontBase = (struct Library *)
      OpenLibrary("diskfont.library", 36L))){
         puts("Unable to open DiskfontBase 36+");
         CloseLibrary((struct Library *)GfxBase);
         CloseLibrary((struct Library *)IntuitionBase);
         IntuitionBase=NULL;   GfxBase=NULL;
         return(FALSE);
   }

   if(!(font = OpenFont(&Topaz80))){
      puts("Cannot Open Topaz 80");
      CloseLibrary((struct Library *)IntuitionBase);
      CloseLibrary((struct Library *)GfxBase);
      CloseLibrary((struct Library *)DiskfontBase);
      IntuitionBase=NULL;  GfxBase=NULL;  DiskfontBase=NULL;
      return(FALSE);
   }

   if(set_pats()) return(TRUE);
   else{
      CloseFont(font);
      CloseLibrary((struct Library *)IntuitionBase);
      CloseLibrary((struct Library *)GfxBase);
      CloseLibrary((struct Library *)DiskfontBase);
      IntuitionBase=NULL;  GfxBase=NULL;  DiskfontBase=NULL;
   }
      return(FALSE);
}
/*----------------------------------------------------*/
/* Must close libraries we opened and free allocated memory before exit */
void Close_Libs(void){
   if(tFont)   CloseFont(tFont);
   if(font) CloseFont(font);
   if(DiskfontBase)  CloseLibrary((struct Library *)DiskfontBase);
   if(IntuitionBase) CloseLibrary((struct Library *)IntuitionBase);
   if(GfxBase) CloseLibrary((struct Library *)GfxBase);
   
   for(i=0;i<10;i++) if(ps_lpat[i])    free(ps_lpat[i]);
   return;
}
/*---------------------------------------------------*/
/* free raster, close window, close screen before exit */
void Close_ScrWin(void){

   if(planePtr)    FreeRaster(planePtr, tmpwidth, tmpheight);
   CloseWindow(sd_win);
   CloseScreen(sd_scr);
   return;
}
/*----------------------------------------------------*/
/* Example that draws a few basic shapes.  Called from
   main() where the Output_Device is set */
void draw_figs(void){
   float x[8], y[8];
   
   /** Draw simple line **/
   sd_Move(MAP_X(25.), MAP_Y(100.));
   sd_Draw(0,1,1,MAP_X(75.),MAP_Y(350.));

   /* draw a patterned, colored line */
   Color_PS=TRUE;    /* first turn color on for PS */
   sd_Move(MAP_X(550.), MAP_Y(350.));
   sd_Draw(3,2,1,MAP_X(450.), MAP_Y(340.));
   
   /* Color_PS=FALSE; */
   
   /* Draw a filled rectangle **/
   sd_Box(MAP_X(300.), MAP_Y(200.), MAP_X(500.), MAP_Y(300.), 1,
      3,0, 1,0,TRUE, TRUE);
   
   /** Select a Font */
   strcpy(F_Name, "CGTimes.font");
   get_font();
   
   /* Draw a Text String */
   sd_Move(MAP_X(500.), MAP_Y(50.));
   sd_Text("Here is TEXT", 3);
   
   /** Select a different font */
   strcpy(F_Name, "CGTriumvirate.font");
   get_font();
   
   /* Draw a Text String */
   sd_Move(MAP_X(350.), MAP_Y(25.));
   sd_Text("Here is Triumvirate TEXT", 1);
    
   /* Draw a polygon */
   x[0]=MAP_X(150.);  y[0]=MAP_Y(200.);
   x[1]=MAP_X(200.);  y[1]=MAP_Y(200.);
   x[2]=MAP_X(300.);  y[2]=MAP_Y(100.);
   x[3]=MAP_X(20.);   y[3]=MAP_Y(20.);
   x[4]=MAP_X(50.);   y[4]=MAP_Y(50.);
   x[5]=MAP_X(150.);  y[5]=MAP_Y(200.);
   sd_AreaDraw2(x,y,6,0,1,0, 2, 0, FALSE, TRUE);    /* without fill */

   /* Draw a polygon */
   x[0]=MAP_X(125.);  y[0]=MAP_Y(125.);
   x[1]=MAP_X(175.);  y[1]=MAP_Y(100.);
   x[2]=MAP_X(160.);  y[2]=MAP_Y(82.);
   x[3]=MAP_X(150.);   y[3]=MAP_Y(100.);
   x[4]=MAP_X(50.);   y[4]=MAP_Y(150.);
   x[5]=MAP_X(125.);  y[5]=MAP_Y(125.);
   sd_AreaDraw2(x,y,6,3,1,0, 2, 0, TRUE, TRUE);    /* WITH fill */

   /* draw a triangle */
   x[0]=MAP_X(400.);  y[0]=MAP_Y(325.);
   x[1]=MAP_X(450.);  y[1]=MAP_Y(275.);
   x[2]=MAP_X(350.);  y[2]=MAP_Y(275.);
   x[3]=MAP_X(400.);  y[3]=MAP_Y(325.);
   sd_AreaDraw2(x,y,4,2,1,0, 2, 0, TRUE, TRUE); /* WITH fill */

   /* draw a patterned, filled, triangle */
   x[0]=MAP_X(450.);  y[0]=MAP_Y(325.);
   x[1]=MAP_X(500.);  y[1]=MAP_Y(275.);
   x[2]=MAP_X(400.);  y[2]=MAP_Y(275.);
   x[3]=MAP_X(450.);  y[3]=MAP_Y(325.);
   sd_AreaDraw2(x,y,4,1,1,3, 2, 0, TRUE, TRUE);

   /** Draw Pie slice, colored fill **/
   sd_DrawSlice2(MAP_X(500.), MAP_Y(75.), 50., 0.0, 72.,1,0, 2, 1,0,TRUE, TRUE);
   
   /** Draw Pie slice, patterned colored fill **/
   sd_DrawSlice2(MAP_X(550.), MAP_Y(100.), 50., 0.0, 72.,1,4, 2, 1,0,TRUE, TRUE);
 
   return;
}

/*************************************************************/
/* Make the font contained in F_Name the current font */
void get_font(void){
    UBYTE *font_buff=NULL;
    int hp_stroke=0, hp_slant=0;
    float correct=1.0;

    tAttr.ta_Name = F_Name;
    tAttr.ta_YSize = F_Size;
    tAttr.ta_Style = F_Style;
    tAttr.ta_Flags = FPF_ROMFONT | FPF_DISKFONT;

    tFont = (struct TextFont *) OpenDiskFont(&tAttr);
    if(!tFont){
        printf("\nCannot Open Font: %s", F_Name);
        /* cannot find user defined font, replace with topaz.font */
        strcpy(F_Name, "topaz.font");
        tAttr.ta_Name = F_Name;
        tFont=(struct TextFont *) OpenDiskFont(&tAttr);
        if(!tFont)  return;
    }
    SetFont(sd_rp, tFont);
    SetSoftStyle(sd_rp, tAttr.ta_Style ^ tFont->tf_Style,
                (FSF_BOLD | FSF_UNDERLINED | FSF_ITALIC));

if(Output_Device == POST){
   if((font_buff = malloc(150 * sizeof(UBYTE))) == NULL)     return;
   font_buff[0] = '\0';
   if((strcmp(F_Name, "CGTimes.font")) == 0){
       if(tFont->tf_Style == 0) strcpy(font_buff,"/Times-Roman");
       else if(tFont->tf_Style == 2)   strcpy(font_buff,"/Times-Bold");
       else if(tFont->tf_Style == 4)   strcpy(font_buff,"/Times-Italic");
       else if(tFont->tf_Style == 6)   strcpy(font_buff,"/Times-BoldItalic");
       else   strcpy(font_buff,"/Times-Roman");
   }
   else if((strcmp(F_Name, "CGTriumvirate.font")) == 0){
       if(tFont->tf_Style == 0)     strcpy(font_buff,"/Helvetica");
       else if(tFont->tf_Style == 2)  strcpy(font_buff,"/Helvetica-Bold");
       else if(tFont->tf_Style == 4)  strcpy(font_buff,"/Helvetica-Oblique");
       else if(tFont->tf_Style == 6)  strcpy(font_buff,"/Helvetica-BoldOblique");
       else   strcpy(font_buff,"/Helvetica");
   }
   else if((strcmp(F_Name, "CGCourier.font")) == 0){
       if(tFont->tf_Style == 0)     strcpy(font_buff,"/Courier");
       else if(tFont->tf_Style == 2)   strcpy(font_buff,"/Courier-Bold");
       else if(tFont->tf_Style == 4)   strcpy(font_buff,"/Courier-Oblique");
       else if(tFont->tf_Style == 6)   strcpy(font_buff,"/Courier-BoldOblique");
       else   strcpy(font_buff,"/Courier");
   }
   else strcpy(font_buff, "/Times-Roman");
   fprintf(Outfp,"%s findfont %.3f scalefont setfont\n",font_buff,(float)F_Size);
}
else if(Output_Device == HPGL2){
    correct = 0.90;  /* very simple correction factor to make font width */
                     /* of HP-GL/2 font similar to CG bullet font width  */
                     
    if(tFont->tf_Style == 0){hp_stroke=0;  hp_slant=0;}
    else if(tFont->tf_Style == 2){hp_stroke=3;  hp_slant=0;}
    else if(tFont->tf_Style == 4){hp_stroke=0;  hp_slant=1;}
    else if(tFont->tf_Style == 6){hp_stroke=3;  hp_slant=1;}
    if((strcmp(F_Name, "CGTimes.font")) == 0){
        fprintf(Outfp,"AD1,277,2,1,4,%.3f,5,%d,6,%d,7,5SA",(float)(F_Size * hp_scalex/10.) * correct, hp_slant,hp_stroke);    /* * correct is to decrease font sz */
    }
    else if((strcmp(F_Name, "CGTriumvirate.font")) == 0){
       fprintf(Outfp,"AD1,277,2,1,4,%.3f,5,%d,6,%d,7,4SA",(float)(F_Size * hp_scalex/10.) * correct, hp_slant,hp_stroke);
    }
    else if((strcmp(F_Name, "CGCourier.font")) == 0){
       fprintf(Outfp,"AD1,277,2,0,4,%.3f,5,%d,6,%d,7,3SA",(float)(F_Size * hp_scalex/10.) * correct,hp_slant,hp_stroke);
    }
    else{    /* use default vector stick font */
       fprintf(Outfp,"AD1,277,2,0,4,%.3f,5,%d,6,%d,7,48SA",(float)(F_Size * hp_scalex/10.) * correct, hp_slant,hp_stroke);
    }
}
else if(Output_Device == DXF){
   /* no DXF font info written out */
}
if(font_buff)  free(font_buff);
return;
}
/*-------------------------------------------------------------*/
/* Draw text to selected output device */
void sd_Text(UBYTE *draw_text, UBYTE draw_color){

   if(Output_Device == SCREEN){
      SetDrMd(sd_rp, JAM2);
      SetAPen(sd_rp, draw_color);
      Text(sd_rp, draw_text, strlen(draw_text));
   }
   else if(Output_Device == POST){
      ps_colorout(draw_color);
      fprintf(Outfp,"(%s) show\n", draw_text);
   }
   else if(Output_Device == HPGL2){
      if(draw_color>0){
         fprintf(Outfp,"SP%dLB%s%c", draw_color, draw_text, 3);
      }
   }
   else if(Output_Device == DXF){
      fprintf(Outfp,"  0\nTEXT\n  8\n2\n 62\n%6d\n 10\n%.4f\n 20\n%.4f\n 40\n%.4f\n",
         draw_color, dxfX * dxfWIDE,
         (8.5-dxfY * dxfHIGH), ((float)(F_Size))/80.); /* 80 should be 72. */
      fprintf(Outfp,"  1\n%s\n  7\nSTANDARD\n 50\n0.0\n 41\n0.6\n",
        draw_text);   
   }
   return;
}
/*--------------------------------------------------------------*/
/* Move drawing pen to specified location */
void sd_Move(float x_loc, float y_loc){
    if(Output_Device == SCREEN){
        Move(sd_rp, (int)(x_loc + 0.5), (int)(y_loc + 0.5));
    }
    else if(Output_Device == POST){
       fprintf(Outfp,"newpath\n%.4f inch %.4f inch moveto\n",
         MAP_PSX(x_loc+0.5),MAP_PSY(y_loc+0.5));
    }
    else if(Output_Device == HPGL2){
        fprintf(Outfp, "PU%.3f,%.3f", x_loc, y_loc);
    }
    else if(Output_Device == DXF){
        dxfX=x_loc;
        dxfY=y_loc;
    }
    return;
}
/*-----------------------------------------------------------*/
/* Draw line from current pen location to specified location */
void sd_Draw(UBYTE pat, UBYTE color, UBYTE l_thick,
            float x_loc, float y_loc)
{
int dp=0;

    if(Output_Device == SCREEN){
        SetAPen(sd_rp, color);
        SetDrPt(sd_rp, lpattern[pat]);
        Draw(sd_rp, (int)(x_loc + 0.5), (int)(y_loc + 0.5));
    }
    else if(Output_Device == POST){
      ps_colorout(color);
      fprintf(Outfp,"%s setdash\n%.3f setlinewidth\n", ps_lpat[pat], l_thick/2.);
      fprintf(Outfp,"%.4f inch %.4f inch lineto\nstroke\n",MAP_PSX(x_loc+0.5), MAP_PSY(y_loc+0.5));
      fputs("[] 0 setdash\n", Outfp);
    }
    else if(Output_Device == HPGL2){
       if(pat==0)  fprintf(Outfp, "LT");
       else  fprintf(Outfp, "LT%d,1.0", pat);
       if(color>0){
           fprintf(Outfp, "SP%dLA2,5PW%.3fPD%.3f,%.3f", color, (double)(l_thick)*0.35,x_loc, y_loc);
       }
    }
    else if(Output_Device == DXF){
        dp=(pat<=8)?pat:(8-pat);
        fprintf(Outfp,"  0\nLINE\n  8\n2\n  6\n%s\n 62\n%6d\n 10\n%f\n 20\n%f\n 11\n%f\n 21\n%f\n",
            DXFLT[dp], color,dxfX * dxfWIDE,(8.5-dxfY * dxfHIGH),
        x_loc * dxfWIDE,(8.5-y_loc * dxfHIGH));
        dxfX=x_loc;
        dxfY=y_loc;
    }
    return;
}
/**************************************************************/
void sd_DrawLine(UBYTE pat, UBYTE color, UBYTE l_thick,
                float x1,float y1,float x2, float y2)
{
int i=0;
int dp=0;

    if(Output_Device == SCREEN){
        SetAPen(sd_rp, color);
        SetDrPt(sd_rp, lpattern[pat]);
        for(i=0;i<l_thick;i++){
            if(x1 != x2){
                Move(sd_rp, (int)(x1+0.5), (int)(y1+0.5+l_thick/2.-i));
                Draw(sd_rp, (int)(x2+0.5), (int)(y2+0.5+l_thick/2.-i));
            }
            else{
                Move(sd_rp, (int)(x1+0.5+l_thick/2.-i), (int)(0.5+y1));
                Draw(sd_rp, (int)(x2+0.5+l_thick/2.-i), (int)(0.5+y2));
            }
        }
    }
    else if(Output_Device == POST){
       ps_colorout(color);
       fprintf(Outfp,"newpath\n%.3f setlinewidth\n%s setdash\n",l_thick/2.,ps_lpat[pat]);
       fprintf(Outfp,"%.4f inch %.4f inch moveto\n%.4f inch %.4f inch lineto\nstroke\n",
          MAP_PSX(x1), MAP_PSY(y1),MAP_PSX(x2), MAP_PSY(y2));
       fputs("[] 0 setdash\n", Outfp);
    }
    else if(Output_Device == HPGL2){
        sd_Move(x1,y1);
        sd_Draw(pat, color, l_thick, x2, y2);
    }
    else if(Output_Device == DXF){
        dp=(pat<=8)?pat:(8-pat);
        fprintf(Outfp,"  0\nLINE\n  8\n2\n  6\n%s\n 62\n%6d\n 10\n%f\n 20\n%f\n 11\n%f\n 21\n%f\n",
            DXFLT[dp], color,x1 * dxfWIDE,(8.5-y1*dxfHIGH),x2 * dxfWIDE,(8.5-y2*dxfHIGH));
        dxfX=x2;
        dxfY=y2;
    }
return;
}
/*----------------------------------------------------------*/
/* Draw a rectangle */
void sd_Box(float x1,float y1,float x2,float y2,int l_thick, int f_col,
   int f_pat, int l_col, int l_pat, BOOL fill, BOOL outline){
    ULONG red=0,green=0,blue=0;
    int i;
    float xmin=0.0,xmax=0.0,ymin=0.0,ymax=0.0;

    if(Output_Device == SCREEN){
        if(x1 < x2){     /* make sure xmin < xmax and ymin < ymax */
            xmin=x1;
            xmax=x2;
        }
        else{
            xmin=x2;
            xmax=x1;
        }
        if(y1 < y2){
            ymin=y1;
            ymax=y2;
        }
        else{
            ymin=y2;
            ymax=y1;
        }
        if(fill){
           SetDrMd(sd_rp, JAM2);
           SetAPen(sd_rp, 0);    /* first fill rect w/ background color */
           SetAfPt(sd_rp, pattern[0], 2);
           BNDRYOFF(sd_rp);
           RectFill(sd_rp, (LONG)(xmin+0.5), (LONG)(ymin+0.5), (LONG)xmax,
            (LONG)ymax);
           SetAPen(sd_rp, f_col);   /* now fill w/ desired color/pattern */
           SetAfPt(sd_rp, pattern[f_pat], 2);
           RectFill(sd_rp, (LONG)(xmin+0.5), (LONG)(ymin+0.5), (LONG)xmax,
            (LONG)ymax);
        }
        if(outline){
           for(i=0;i<l_thick;i++){
               sd_Move((xmin+i), (ymin+i));
               sd_Draw(l_pat,l_col,1, (xmax-i), (ymin+i));
               sd_Draw(l_pat,l_col,1, (xmax-i), (ymax-i));
               sd_Draw(l_pat,l_col,1, (xmin+i), (ymax-i));
               sd_Draw(l_pat,l_col,1, (xmin+i), (ymin+i));
           }
       }
    }
    else if(Output_Device == POST){
    /************ set inside of box to background color **********/
    if(fill){
       ps_colorout(0);
       fprintf(Outfp,"%.4f inch %.4f inch moveto\n%.4f inch %.4f inch lineto\n",
          MAP_PSX(x1), MAP_PSY(y1),MAP_PSX(x2), MAP_PSY(y1));
       fprintf(Outfp,
          "%.4f inch %.4f inch lineto\n%.4f inch %.4f inch lineto\nclosepath\nfill\n",
          MAP_PSX(x2), MAP_PSY(y2),MAP_PSX(x1), MAP_PSY(y2));

    /************* now fill box with desired color or pattern ************/
       ps_colorout(f_col);
        fprintf(Outfp,"%.3f setlinewidth\n",l_thick/2.);
        fprintf(Outfp,"%.4f inch %.4f inch moveto\n%.4f inch %.4f inch lineto\n",
            MAP_PSX(x1), MAP_PSY(y1),MAP_PSX(x2), MAP_PSY(y1));
        fprintf(Outfp,
            "%.4f inch %.4f inch lineto\n%.4f inch %.4f inch lineto\nclosepath\n",
            MAP_PSX(x2), MAP_PSY(y2),MAP_PSX(x1), MAP_PSY(y2));

/** fill with a pattern **/
        fprintf(Outfp,"gsave\nclip\n%s\ngrestore\n", Pattern_PS[f_pat]);
      }
      if(outline){
/******* create the box outline *******/
        if(Color_PS){
            split_colors(l_col, &red, &green, &blue);
            fprintf(Outfp,"newpath\n%.4f %.4f %.4f setrgbcolor\n",red/15.,green/15.,blue/15.);
        }
        else
            fprintf(Outfp,"newpath\n%.4f setgray\n", ((GetRGB4(ssgcm, l_col))/4095.));

        fprintf(Outfp,"%.4f setlinewidth\n", l_thick/2.);
        fprintf(Outfp,"%.4f inch %.4f inch moveto\n%.4f inch %.4f inch lineto\n",
            MAP_PSX(x1), MAP_PSY(y1),MAP_PSX(x2), MAP_PSY(y1));
        fprintf(Outfp,"%.4f inch %.4f inch lineto\n%.4f inch %.4f inch lineto\nclosepath\nstroke\n",
            MAP_PSX(x2), MAP_PSY(y2),MAP_PSX(x1), MAP_PSY(y2));
      }
    }
    else if(Output_Device == HPGL2){
       hp_fillpat(f_col, f_pat);
       if(f_col>0 && fill){     /** fill rectangle **/
          fprintf(Outfp, "PU%.3f,%.3fRA%.3f,%.3f", x1,y1,x2,y2);
       }
       /*   outline rectangle */
       if(outline)  fprintf(Outfp, "PW%.3fSP1PU%.3f,%.3fEA%.3f,%.3f",
         ((float)(l_thick)*0.35), x1, y1, x2, y2);
    }
    else if(Output_Device == DXF){  /* code only creates wireframe */
        sd_Move(x1, y1);
        sd_Draw(0,l_col,l_thick,x2, y1);
        sd_Draw(0,l_col,l_thick,x2, y2);
        sd_Draw(0,l_col,l_thick,x1, y2);
        sd_Draw(0,l_col,l_thick,x1, y1);
        dxfX=x1;
        dxfY=y1;
    }
    return;
}
/*--------------------------------------------------------------------*/
/** Used to draw pie slice **/
BOOL sd_DrawSlice2(float xloc, float yloc, float rad, float angstart,
        float angend, int l_thick, int f_pat, int f_col,
        int l_col, int l_pat, BOOL fill, BOOL outline){
    register int i=0;
    float aspect, xrad=0.0, yrad=0.0, *xpie=NULL, *ypie=NULL;
    float xtemp=0.0, ytemp=0.0, temp=0.0, endang, incr = 1.0;
    float rad1, rad2;   /* used by PS code to switch x and y radius if landscape */

   aspect = 0.82;
   temp = floor(angstart);
   endang = floor(angend);
   if(endang==359.) endang=360.0;   /* correct factor */
   if(Output_Device == SCREEN || Output_Device==HPGL2 || Output_Device == DXF){
      if(!(xpie=malloc(732 * sizeof(float))))   return(FALSE);
      if(!(ypie=malloc(732 * sizeof(float)))){
         free(xpie);
         return(FALSE);
      }
      loc_pieline(rad, temp, aspect, &xtemp, &ytemp);
      xpie[0]=xloc;ypie[0]=yloc;xpie[1]=xloc+xtemp;ypie[1]=yloc+ytemp;
      i=2;
      while(temp<=endang && i<730){
         loc_pieline(rad, temp, aspect, &xtemp, &ytemp);
         xpie[i]=xloc+xtemp;
         ypie[i]=yloc+ytemp;
         temp += incr;
         xtemp=ytemp=0.0;
         i++;
      }
      xpie[i]=xpie[0];
      ypie[i]=ypie[0];
      i++;
      sd_AreaDraw2(xpie, ypie, i, f_col, l_col, f_pat, l_thick, l_pat, 
         fill, outline);
   }
   else if(Output_Device == POST){
      xrad = rad;
      yrad=xrad;
      rad1 = MAP_PSX2(xrad);
      rad2 = MAP_PSX2(yrad);
      ps_colorout(0);
      /** this code clears underlying garbage by filling with color 0 **/
      fprintf(Outfp,"newpath\n%.3f inch %.3f inch %.3f inch %.3f inch %.3f %.3f ellipse\n",
         MAP_PSX(xloc), MAP_PSY(yloc), rad1, rad2, angstart, angend);
      fprintf(Outfp,"%.3f inch %.3f inch lineto\nclosepath\nfill\n",
         MAP_PSX(xloc),MAP_PSY(yloc));
      if(fill){
         ps_colorout(f_col);   /** select color or grayscale ***/
/******* first fill the slice, remember to setgray ************/
         fprintf(Outfp,"%.3f setlinewidth\n",(double)(l_thick)/2.);
         fprintf(Outfp,"%.3f inch %.3f inch %.3f inch %.3f inch %.3f %.3f ellipse\n",
            MAP_PSX(xloc), MAP_PSY(yloc), rad1, rad2, angstart, angend);
         fprintf(Outfp,"%.3f inch %.3f inch lineto\nclosepath\n",
            MAP_PSX(xloc),MAP_PSY(yloc));
         ps_fillout(f_pat);
      }
      if(outline){
/******* now stroke the slice outline, remember to setgray ************/
         fprintf(Outfp,"newpath\n%.3f setlinewidth\n",(double)(l_thick)/2.);
         ps_colorout(l_col);
         fprintf(Outfp,"%.3f inch %.3f inch %.3f inch %.3f inch %.3f %.3f ellipse\n",
            MAP_PSX(xloc),MAP_PSY(yloc), rad1, rad2, angstart, angend);
         fprintf(Outfp,"%.3f inch %.3f inch lineto\nclosepath\nstroke\n",
            MAP_PSX(xloc),MAP_PSY(yloc));
      }
  }
return(TRUE);
}
/*------------------------------------------------------------------*/
/* Draws an arc, line only, no fill */
void arc(float xloc, float yloc, float rad, float ang1, float ang2,
   int l_thick, int l_col, int l_pat){

   register int i=0;
   float aspect, xt=0.0, yt=0.0, tmp=0.0, endang, incr = 1.0;      
   
   aspect = 0.82;
   tmp = floor(ang1);
   endang = floor(ang2);
   if(endang==359.) endang=360.0;   /* correct factor */
   loc_pieline(rad, tmp, aspect, &xt, &yt);
   sd_Move(xloc+xt, yloc+yt);
   i=1;
   while(tmp<=endang && i<730){
      loc_pieline(rad, tmp, aspect, &xt, &yt);
      sd_Draw(l_pat, l_col, l_thick, xloc+xt, yloc+yt);
      tmp += incr;
      i++;
   }
   return;
}
/*------------------------------------------------------------------*/
/* prints out color or grayscale info for PS graph */
void ps_colorout(ULONG d_color){
   ULONG red=0,green=0,blue=0;
   float gray_level=0.0;
      
   if(Color_PS){
      split_colors(d_color, &red, &green, &blue);
      fprintf(Outfp, "%.4f %.4f %.4f setrgbcolor\n",red/15.,green/15.,blue/15.);
   }
   else{
      gray_level = (GetRGB4(ssgcm, d_color))/4095.;
      fprintf(Outfp, "%.4f setgray\n", gray_level);
   }
   return;
}

/*------------------------------------------------------------------*/
/* prints out pattern or fill info for PS graph */
void ps_fillout(int d_pat){
   fprintf(Outfp,"gsave\nclip\n%s\ngrestore\n",Pattern_PS[d_pat]);
   return;
}
/*---------------------------------------------------------------------*/
/* calculates locations of pie line, angle is in degrees **/
/* For non-FPU systems, may want to construct lookup-table */
void loc_pieline(float radius, float angle, float aspect, float *xt, float *yt){
    if(aspect < 1.0){
        *xt = (float)cos(angle * DTOR) * radius;
        *yt = -aspect * (float)sin(angle * DTOR) * radius;
    }
    else{
        *xt = (float)cos(angle * DTOR) * radius/aspect;
        *yt = -((float)sin(angle * DTOR) * radius);
    }
    return;
}
/**----------------------------------------------------------------------**/
/**   Draws a closed polygon, fills it if desired, and outlines it if desired.
      x and y are in screen coordinates 
      Note, before can do AmigaDos AreaDraw, must allocate areaBuffer.
**/
BOOL sd_AreaDraw2(float *x, float *y, int pt_sz, int f_col, int l_col, int f_pat,
   int l_thick, int l_pat, BOOL fill, BOOL outline){
   register int i;
   UWORD *areaBuffer=NULL;
   int j=0;
   
   if(Output_Device==SCREEN){
      if(fill){
         if(!(areaBuffer = sg_AreaBuffer(pt_sz+2))){
            printf("\nUnable to Allocate Memory!");
            return(FALSE);
         }
         SetDrMd(sd_rp, JAM2);     
         SetAPen(sd_rp, f_col);
         SetAfPt(sd_rp, pattern[f_pat], 2);
         BNDRYOFF(sd_rp);
         AreaMove(sd_rp, (int)(x[0]+0.5), (int)(y[0]+0.5));
         for(i=1;i<pt_sz;i++)
             AreaDraw(sd_rp, (int)(x[i]+0.5), (int)(y[i]+0.5));
         AreaEnd(sd_rp);
         free(areaBuffer);
      }
      if(outline){
         sd_Move(x[0], y[0]);
         for(i=0;i<pt_sz;i++)
            sd_Draw(l_pat, l_col, l_thick, (float)x[i], (float)y[i]);
      }
   }
    
   else if(Output_Device == POST){
      if(fill){
         /** first clear background under shape **/
         fprintf(Outfp,"newpath\n%.3f inch %.3f inch moveto\n",
            MAP_PSX(x[0]), MAP_PSY(y[0]));
         for(j=1;j<pt_sz;j++){
            fprintf(Outfp,"%f inch %f inch lineto\n",
               MAP_PSX(x[j]), MAP_PSY(y[j]));
         }
         fputs("closepath\ngsave\n", Outfp);
         ps_colorout(0);
         fputs("fill\ngrestore\n", Outfp);
      
         /** now fill shape with desired pattern **/
         ps_colorout(f_col);
         ps_fillout(f_pat);
      }
      if(outline){
         ps_colorout(l_col);
         fprintf(Outfp,"%.3f setlinewidth\n", l_thick/2.0);
         fprintf(Outfp,"newpath\n%.3f inch %.3f inch moveto\n",
            MAP_PSX(x[0]), MAP_PSY(y[0]));
         for(j=1;j<pt_sz;j++){
            fprintf(Outfp,"%f inch %f inch lineto\n",
               MAP_PSX(x[j]), MAP_PSY(y[j]));
         }      
         fputs("closepath\nstroke\n", Outfp);
      }
   }
    
    else if(Output_Device==HPGL2){
        sd_Move(x[0], y[0]);
        fprintf(Outfp,"PM");
        for(i=0;i<pt_sz;i++)
            fprintf(Outfp, "PD%.3f,%.3f", x[i], y[i]);
        fprintf(Outfp, "PM2");   /* may have erased some of the code */
        if(fill && f_col>0){
            hp_fillpat(f_col, f_pat);
            fprintf(Outfp, "FP");
        }
        if(outline){
            if(l_pat>0) fprintf(Outfp, "LT%dPW%.3fSP%dEP", l_pat,
               (float)l_thick * 0.35, l_col);
            else  fprintf(Outfp, "LTPW%.3fSP%dEP",(float)l_thick*0.35,l_col);
        }
    }
    else if(Output_Device==DXF){    /* wireframe only */
        sd_Move(x[0], y[0]);
        for(i=1;i<pt_sz;i++)    sd_Draw(0,1,1, x[i], y[i]);
    }
    return(TRUE);
}
/*-----------------------------------------------------------*/
/**   Allocates memory for vertices of polygon drawn using sg_AreaDraw **/
UWORD *sg_AreaBuffer(int pt_sz){
   register int i;
   int num;
   UWORD *p=NULL;
   
    num = pt_sz * 5;
    if(!(p = malloc(num * sizeof(UWORD))))   return(NULL);
    for(i=0;i<num; i++)     p[i] = 0;
    InitArea(&areaInfo, p, (pt_sz * 2));
    sd_rp->AreaInfo = &areaInfo;
    return(p);
}
/*------------------------------------------------------------------------------*/
void split_colors(UBYTE color, ULONG *red, ULONG *green, ULONG *blue)
{
ULONG cmap_value=0;

cmap_value = GetRGB4(ssgcm, color);
*red = ((cmap_value & 0x0f00) >> 8);
*green = ((cmap_value & 0x00f0) >> 4);
*blue = ((cmap_value & 0x000f));

return;
}
/*----------------------------------------------------------*/
BOOL set_pats(void){
pattern[0][0] = 0xFFFF;    /* used to be 0x0000 */
pattern[0][1] = 0xFFFF;
pattern[0][2] = 0xFFFF;
pattern[0][3] = 0xFFFF;

pattern[1][0] = 0x0000;
pattern[1][1] = 0xFFFF;
pattern[1][2] = 0x0000;
pattern[1][3] = 0x0000;

pattern[2][0] = 0xAAAA;
pattern[2][1] = 0xAAAA;
pattern[2][2] = 0xAAAA;
pattern[2][3] = 0xAAAA;

pattern[3][0] = 0xC0C0;
pattern[3][1] = 0x0C0C;
pattern[3][2] = 0xC0C0;
pattern[3][3] = 0x0C0C;

pattern[4][0] = 0xCCCC;
pattern[4][1] = 0xCCCC;
pattern[4][2] = 0xCCCC;
pattern[4][3] = 0xCCCC;

pattern[5][0] = 0xCCCC;
pattern[5][1] = 0x3333;
pattern[5][2] = 0xCCCC;
pattern[5][3] = 0x3333;

pattern[6][0] = 0x8888;
pattern[6][1] = 0x8888;
pattern[6][2] = 0x8888;
pattern[6][3] = 0x8888;

pattern[7][0] = 0x0000;
pattern[7][1] = 0xF0F0;
pattern[7][2] = 0x0000;
pattern[7][3] = 0xF0F0;

pattern[8][0] = 0x2222;
pattern[8][1] = 0x8888;
pattern[8][2] = 0x1111;
pattern[8][3] = 0x4444;
/********** set pattern for lines **********/

lpattern[0] = 0xFFFF;       /** 1111111111111111  ..............  **/
lpattern[1] = 0xAAAA;       /** 1010101010101010  . . . . . . . **/
lpattern[2] = 0xCCCC;       /** 1100110011001100  ..  ..  ..  ..**/
lpattern[3] = 0x8888;       /** 1000100010001000  .   .   .   . **/
lpattern[4] = 0xF0F0;       /** 1111000011110000  ....    ....    .... **/
lpattern[5] = 0xF6F6;       /** 1111011011110110  .... .. .... .. ....**/
lpattern[6] = 0xEAEA;       /** 1110101011101010  ... . . ... . . ... **/
lpattern[7] = 0xF22F;       /** 1111001000101111  ....  .   . ....  **/
lpattern[8] = 0xE667;       /** 1110011001100111  ...  ..  ..  ...**/

for(i=0;i<10;i++) if((ps_lpat[i] = malloc(20 * sizeof(UBYTE))) == NULL)     return(FALSE);

/** line patterns for PostScript **/
strcpy(ps_lpat[0], "[] 0 ");
strcpy(ps_lpat[1], "[2 2] 0 ");
strcpy(ps_lpat[2], "[4 4] 0 ");
strcpy(ps_lpat[3], "[2 6] 0 ");
strcpy(ps_lpat[4], "[8 8] 0 ");
strcpy(ps_lpat[5], "[4 2 2 2] 0 ");
strcpy(ps_lpat[6], "[1 2] 0 ");
strcpy(ps_lpat[7], "[4 2 2 2 2 2] 0 ");
strcpy(ps_lpat[8], "[8 2 4 2 4 8] 0 ");

/** fill patterns for PostScript **/
strcpy(Pattern_PS[0], "fill");
strcpy(Pattern_PS[1], "sdpat1");
strcpy(Pattern_PS[2], "sdpat2");
strcpy(Pattern_PS[3], "sdpat3");
strcpy(Pattern_PS[4], "sdpat4");
strcpy(Pattern_PS[5], "sdpat5");
strcpy(Pattern_PS[6], "sdpat6");
strcpy(Pattern_PS[7], "sdpat7");
strcpy(Pattern_PS[8], "sdpat8");
return(TRUE);
}
/*------------------------------------------------------*/
/**   This function outputs fill pattern instructions for HP-GL/2 plots.
**/
void hp_fillpat(UBYTE color, UBYTE pattern){
   UBYTE buff[7];
   
   buff[0]='\0';
   strcpy(buff, "PW");
   
   if(Outfp && color > 0){   /** make sure file pointer open and color not white **/
      fprintf(Outfp,"SP%d", color);
      switch(pattern){
         case 0:
         fprintf(Outfp, "%sLTFT1", buff);  /**Should be solid fill **/
         break;
         
         case 1:
         fprintf(Outfp, "%sLTFT3,6,0", buff);
         break;
         
         case 2:
         fprintf(Outfp, "%sLTFT3,4,90", buff);
         break;

         case 3:
         fprintf(Outfp, "%sLT2FT3,4,90", buff);
         break;

         case 4:
         fprintf(Outfp, "%sLTFT3,10,90", buff);
         break;
         
         case 5:
         fprintf(Outfp, "%sLTFT4,6,45", buff);  /** cross hatch **/
         break;
         
         case 6:
         fprintf(Outfp, "%sLTFT3,3,60", buff);
         break;
         
         case 7:
         fprintf(Outfp, "%sLT2,1.0FT3,6,90", buff);
         break;
         
         case 8:
         fprintf(Outfp, "%sLT2,1.0FT4,8,45", buff);   /** cross hatched dashed **/
         break;
      }
   }
   return;
}
